<?php
if (!defined('ABSPATH')) {
	exit;
}

final class ACWPCommentProtection {
	// stara opcja (zostawiamy, żeby nie tracić ustawień po migracji z motywu).
	public const OPTION_BANNED_TERMS = 'theme_comment_protection_banned_terms';

	// nowe nazwy logiczne, ale HARD mapujemy na starą opcję.
	public const OPTION_HARD_TERMS = self::OPTION_BANNED_TERMS;
	public const OPTION_SOFT_TERMS = 'theme_comment_protection_soft_terms';
	public const OPTION_BLOCKED_EMAIL_SUFFIXES = 'theme_comment_protection_blocked_email_suffixes';

	// query var do komunikatów dla trybu bez JS / ominięcie formularza.
	private const NOTICE_QUERY_VAR = 'cp_notice';

	public static function init(): void {
		// email opcjonalny (tylko frontend).
		add_filter('pre_option_require_name_email', [__CLASS__, 'disableCoreRequireNameEmail'], 999);

		// twarda blokada przy klasycznym submit (wp-comments-post.php),
		// żeby komentarz NIE został zapisany w WP.
		add_action('pre_comment_on_post', [__CLASS__, 'hardBlockClassicSubmission'], 0);

		// komunikat nad formularzem (działa też bez JS).
		add_action('comment_form_before', [__CLASS__, 'renderNoticeFromQuery'], 1);
	}

	public static function disableCoreRequireNameEmail($value) {
		// w panelu admina zostawiamy domyślne zachowanie.
		if (is_admin()) return $value;
		return 0;
	}

	public static function shouldBypassAllFilters(): bool {
		// zalogowani zwykle są „zaufani” (np. Ty / redaktorzy).
		return is_user_logged_in();
	}

	/**
	 * Komentarz po polsku:
	 * Minimalne rozróżnienie powodów odrzucenia:
	 * - 'url'  => link / adres URL / honeypot url
	 * - 'spam' => pozostałe przypadki
	 */
	public static function getBlockReasonForGuest(array $postData): string {
		// Honeypot: "Witryna" (url) wypełniona = traktujemy jako URL.
		$url = trim((string)($postData['url'] ?? ''));
		if ($url !== '') {
			return 'url';
		}

		$content = (string)($postData['comment'] ?? $postData['comment_content'] ?? '');
		if ($content !== '' && self::hasAnyLinkAggressive($content)) {
			return 'url';
		}

		return 'spam';
	}

	/**
	 * Komentarz po polsku:
	 * Bezpiecznik dla klasycznego submitu (nie AJAX).
	 * Jeśli komentarz ma zostać odrzucony → redirect do referera z ?cp_notice=url|spam,
	 * żeby można było wyświetlić komunikat z klasą CSS.
	 */
	public static function hardBlockClassicSubmission(): void {
		if (self::shouldBypassAllFilters()) return;

		// nie psujemy odpowiedzi AJAX (tam walidacja jest w handlerze).
		if (function_exists('wp_doing_ajax') && wp_doing_ajax()) return;

		$post = wp_unslash($_POST);

		if (self::isAllowedForGuest($post)) {
			return;
		}

		$reason = self::getBlockReasonForGuest($post); // 'url' lub 'spam'
		$ref = wp_get_referer();

		// preferujemy powrót na stronę wpisu (referer z formularza).
		if (is_string($ref) && $ref !== '') {
			$target = remove_query_arg(self::NOTICE_QUERY_VAR, $ref);
			$target = add_query_arg(self::NOTICE_QUERY_VAR, $reason, $target);
			wp_safe_redirect($target, 302);
			exit;
		}

		// fallback jeśli brak referera (rzadkie przypadki botów).
		$msg = ($reason === 'url')
			? 'Nasze systemy wykryły link w komentarzu i został odrzucony.'
			: 'Nasze systemy antyspamowe wykryły podejrzaną treść i komentarz został odrzucony.';

		wp_die(esc_html($msg), '', ['response' => 200]);
	}

	/**
	 * Komentarz po polsku:
	 * Wyświetla komunikat nad formularzem na podstawie ?cp_notice=...
	 * Klasy:
	 * - comment-protection-notice--success
	 * - comment-protection-notice--error
	 * - comment-protection-notice--url
	 */
	public static function renderNoticeFromQuery(): void {
		if (self::shouldBypassAllFilters()) return;

		$type = isset($_GET[self::NOTICE_QUERY_VAR]) ? (string)wp_unslash($_GET[self::NOTICE_QUERY_VAR]) : '';
		$type = trim($type);
		if ($type === '') return;

		// dopuszczamy też "success" pod przyszłe scenariusze.
		$allowed = ['success', 'spam', 'url'];
		if (!in_array($type, $allowed, true)) return;

		$message = '';
		$variantClass = '';
		$role = 'alert';

		if ($type === 'success') {
			$message = 'Komentarz dodany i niedługo pojawi się na stronie';
			$variantClass = 'comment-protection-notice--success';
			$role = 'status';
		} elseif ($type === 'url') {
			$message = 'Nasze systemy wykryły link w komentarzu i został odrzucony.';
			$variantClass = 'comment-protection-notice--url';
			$role = 'alert';
		} else { // spam
			$message = 'Nasze systemy antyspamowe wykryły podejrzaną treść i komentarz został odrzucony.';
			$variantClass = 'comment-protection-notice--error';
			$role = 'alert';
		}

		echo '<div class="comment-protection-notice ' . esc_attr($variantClass) . '" role="' . esc_attr($role) . '" tabindex="-1">'
			. esc_html($message)
			. '</div>';
	}

	public static function isAllowedForGuest(array $postData): bool {
		// Honeypot: "Witryna" (url) wypełniona = odrzut.
		$url = trim((string)($postData['url'] ?? ''));
		if ($url !== '') return false;

		$content = (string)($postData['comment'] ?? $postData['comment_content'] ?? '');
		if (trim($content) === '') return false;

		// 1) NAJWAŻNIEJSZE: jakikolwiek link/adres URL w treści → odrzut dla niezalogowanych.
		if (self::hasAnyLinkAggressive($content)) {
			return false;
		}

		// 2) Blokada domen e-mail (np. .ru; mail.ru; .com).
		$email = (string)($postData['email'] ?? $postData['comment_author_email'] ?? '');
		if ($email !== '' && self::isEmailBlocked($email)) {
			return false;
		}

		// 3) HARD (agresywne dopasowanie „po zgniataniu”).
		$hardTerms = self::getHardTerms();
		if (!empty($hardTerms)) {
			$author = (string)($postData['author'] ?? $postData['comment_author'] ?? '');
			$scanHard = self::normalizeForHardScan($author . ' ' . $email . ' ' . $content);

			foreach ($hardTerms as $term) {
				$term = trim((string)$term);
				if ($term === '') continue;

				if (self::hardTermMatches($scanHard, $term)) {
					return false;
				}
			}
		}

		// 4) SOFT (dotychczasowa logika dopasowania).
		$softTerms = self::getSoftTerms();
		if (!empty($softTerms)) {
			$author = (string)($postData['author'] ?? $postData['comment_author'] ?? '');
			$scan = self::normalizeForScan($author . ' ' . $email . ' ' . $content);

			foreach ($softTerms as $term) {
				$term = trim((string)$term);
				if ($term === '') continue;

				$t = self::normalizeForScan($term);
				if (mb_strlen($t) < 2) continue;

				if (self::softTermMatches($scan, $t)) {
					return false;
				}
			}
		}

		// 5) (opcjonalnie) reguła „krótki hejt bez uzasadnienia”.
		if (self::isLowValueNegativeComment($content)) {
			return false;
		}

		return true;
	}

	public static function getHardTerms(): array {
		return self::getOptionTerms(self::OPTION_HARD_TERMS);
	}

	public static function getSoftTerms(): array {
		return self::getOptionTerms(self::OPTION_SOFT_TERMS);
	}

	private static function getOptionTerms(string $optionName): array {
		$raw = trim((string)get_option($optionName, ''));
		if ($raw === '') return [];

		$parts = array_map('trim', explode(';', $raw));
		$parts = array_filter($parts, static function ($w) { return $w !== ''; });

		$unique = [];
		foreach ($parts as $w) {
			$unique[mb_strtolower($w)] = $w;
		}

		return array_values($unique);
	}

	public static function getBlockedEmailSuffixes(): array {
		$raw = trim((string)get_option(self::OPTION_BLOCKED_EMAIL_SUFFIXES, ''));
		if ($raw === '') return [];

		$parts = array_map('trim', explode(';', $raw));
		$parts = array_filter($parts, static function ($w) { return $w !== ''; });

		$unique = [];
		foreach ($parts as $w) {
			$w = mb_strtolower(trim($w));
			$w = ltrim($w, "@ \t\r\n");
			$w = trim($w);
			if ($w === '') continue;
			if (mb_strlen($w) < 2) continue;
			$unique[$w] = $w;
		}

		return array_values($unique);
	}

	private static function isEmailBlocked(string $email): bool {
		$email = mb_strtolower(trim($email));
		if ($email === '' || mb_strpos($email, '@') === false) return false;

		$parts = explode('@', $email);
		$domain = trim((string)end($parts));
		if ($domain === '') return false;

		$blocked = self::getBlockedEmailSuffixes();
		if (empty($blocked)) return false;

		foreach ($blocked as $suffix) {
			$suffix = mb_strtolower(trim((string)$suffix));
			if ($suffix === '') continue;
			if (self::strEndsWith($domain, $suffix)) return true;
		}

		return false;
	}

	private static function strEndsWith(string $haystack, string $needle): bool {
		$len = mb_strlen($needle);
		if ($len === 0) return true;
		return (mb_substr($haystack, -$len) === $needle);
	}

	public static function normalizeForScan(string $text): string {
		$text = wp_strip_all_tags($text);
		$text = html_entity_decode($text, ENT_QUOTES | ENT_HTML5, 'UTF-8');
		$text = mb_strtolower($text);
		$text = preg_replace('/[\s]+/u', ' ', $text);
		$text = str_replace(["\u{00A0}", "\r", "\n", "\t"], ' ', $text);
		return trim((string)$text);
	}

	public static function normalizeForHardScan(string $text): string {
		$text = self::normalizeForScan($text);
		// obfuscation.
		$text = str_replace(['hxxps', 'hxxp'], ['https', 'http'], $text);
		// zgniatamy separatory.
		$text = preg_replace('~[^\p{L}\p{N}]+~u', '', $text);
		return (string)$text;
	}

	private static function hardTermMatches(string $scanHard, string $term): bool {
		$termNorm = self::normalizeForHardScan($term);
		if ($termNorm === '' || mb_strlen($termNorm) < 3) return false;
		return (mb_strpos($scanHard, $termNorm) !== false);
	}

	/**
	 * Komentarz po polsku:
	 * Twarde wykrywanie linków/adresów URL w treści komentarza.
	 * Łapie:
	 * - [url=...]...[/url]
	 * - https://...
	 * - domeny bez protokołu (example.com, sub.domain.ru/path)
	 * - IP
	 * - href=
	 * - obfuscation: h t t p : / /, w w w .
	 */
	public static function hasAnyLinkAggressive(string $content): bool {
		if ($content === '') return false;

		// 1) WP helper - wyciąga typowe URL-e.
		if (function_exists('wp_extract_urls')) {
			$urls = wp_extract_urls($content);
			if (!empty($urls)) return true;
		}

		// 2) HTML href.
		if (preg_match('~<a\s[^>]*href\s*=~i', $content)) return true;

		// 3) BBCode [url=...].
		if (preg_match('~\[\s*url\s*=\s*[^\]]+\]~iu', $content)) return true;
		if (preg_match('~\[\s*/\s*url\s*\]~iu', $content)) return true;

		$raw = wp_strip_all_tags($content);
		$raw = html_entity_decode($raw, ENT_QUOTES | ENT_HTML5, 'UTF-8');
		$rawLower = mb_strtolower($raw);

		// 4) protokoły / www.
		if (preg_match('~\b(?:https?|ftp)://~i', $rawLower)) return true;
		if (preg_match('~\bwww\.~i', $rawLower)) return true;
		if (preg_match('~\bmailto:~i', $rawLower)) return true;

		// 5) obfuscation: h t t p : / /  i  w w w .
		if (preg_match('~h\W*t\W*t\W*p\W*s?\W*:\W*/\W*/~iu', $rawLower)) return true;
		if (preg_match('~w\W*w\W*w\W*\W*\.~iu', $rawLower)) return true;

		// 6) IP.
		if (preg_match('~\b(?:(?:25[0-5]|2[0-4]\d|1?\d?\d)\.){3}(?:25[0-5]|2[0-4]\d|1?\d?\d)\b~', $rawLower)) {
			return true;
		}

		// 7) Domeny ASCII/punycode + opcjonalna ścieżka.
		if (preg_match('~\b(?:[a-z0-9](?:[a-z0-9-]{0,61}[a-z0-9])?\.)+(?:[a-z]{2,24}|xn--[a-z0-9-]{2,})\b(?:/[^\s<]*)?~i', $rawLower)) {
			return true;
		}

		// 8) Domeny z Unicode (IDN w formie natywnej).
		if (preg_match('~\b(?:[\p{L}\p{N}](?:[\p{L}\p{N}-]{0,61}[\p{L}\p{N}])?\.)+(?:[\p{L}]{2,24})\b(?:/[^\s<]*)?~iu', $rawLower)) {
			return true;
		}

		// 9) Zbity skan - łapie ekstremalne separatory.
		$compact = preg_replace('~[^\p{L}\p{N}]+~u', '', $rawLower);
		if (!$compact) return false;
		if (mb_strpos($compact, 'http') !== false) return true;
		if (mb_strpos($compact, 'www') !== false) return true;

		return false;
	}

	private static function softTermMatches(string $scan, string $termNorm): bool {
		$termNorm = trim($termNorm);
		if ($termNorm === '') return false;

		if ($termNorm === '@') {
			return (mb_strpos($scan, '@') !== false);
		}

		if (!preg_match('~[\p{L}\p{N}]~u', $termNorm)) {
			return false;
		}

		if (mb_strpos($termNorm, ' ') !== false) {
			$escaped = preg_quote($termNorm, '~');
			$escaped = preg_replace('~\\\s+~u', '\\s+', $escaped);
			$pattern = '~(?<![\p{L}\p{N}])' . $escaped . '(?![\p{L}\p{N}])~u';
			return (bool)preg_match($pattern, $scan);
		}

		if (substr($termNorm, -1) === '*') {
			$base = trim(rtrim($termNorm, '*'));
			if ($base === '') return false;

			if (!preg_match('~^[\p{L}\p{N}]+$~u', $base)) {
				return (mb_strpos($scan, $base) !== false);
			}

			$pattern = '~(?<![\p{L}\p{N}])' . preg_quote($base, '~') . '[\p{L}\p{N}]*~u';
			if (preg_match($pattern, $scan)) return true;

			if (mb_strlen($base) >= 5) {
				$sep = self::buildSeparatedCharsPattern($base);
				$pattern2 = '~(?<![\p{L}\p{N}])' . $sep . '[\p{L}\p{N}]*~u';
				if (preg_match($pattern2, $scan)) return true;
			}

			return false;
		}

		if (!preg_match('~^[\p{L}\p{N}]+$~u', $termNorm)) {
			return (mb_strpos($scan, $termNorm) !== false);
		}

		$pattern = '~(?<![\p{L}\p{N}])' . preg_quote($termNorm, '~') . '(?![\p{L}\p{N}])~u';
		if (preg_match($pattern, $scan)) return true;

		if (mb_strlen($termNorm) >= 5) {
			$sep = self::buildSeparatedCharsPattern($termNorm);
			$pattern2 = '~(?<![\p{L}\p{N}])' . $sep . '(?![\p{L}\p{N}])~u';
			if (preg_match($pattern2, $scan)) return true;
		}

		return false;
	}

	private static function buildSeparatedCharsPattern(string $word): string {
		$chars = preg_split('//u', $word, -1, PREG_SPLIT_NO_EMPTY);
		if (!$chars) return preg_quote($word, '~');

		$parts = [];
		foreach ($chars as $ch) {
			$parts[] = preg_quote($ch, '~');
		}

		return implode('[^\p{L}\p{N}]*', $parts);
	}

	private static function isLowValueNegativeComment(string $content): bool {
		$norm = self::normalizeForScan($content);
		if ($norm === '') return false;

		$charCount = mb_strlen($norm);
		$words = preg_split('/\s+/u', trim($norm)) ?: [];
		$wordCount = count(array_filter($words, static function ($w) { return $w !== ''; }));

		$isShort = ($charCount <= 160) || ($wordCount <= 22);
		if (!$isShort) return false;

		$reasonMarkers = [
			'bo ', 'ponieważ', 'poniewaz', 'dlatego', 'przez to', 'wynika', 'z powodu', 'problem', 'problemy',
			'cena', 'cen', 'aktualn', 'parametr', 'funkcj', 'działa', 'dziala', 'test', 'porówn', 'porown',
			'brak', 'nie ma', 'minus', 'wada', 'wady', 'zalet', 'plus',
		];

		foreach ($reasonMarkers as $r) {
			if ($r !== '' && mb_strpos($norm, $r) !== false) {
				return false;
			}
		}

		$negativeMarkers = [
			'słabe', 'slabe', 'słaby', 'slaby', 'kiepsk', 'fatal', 'tragic', 'porażk', 'porazk',
			'bez sensu', 'bzdur', 'głupot', 'glupot', 'żałos', 'zalos', 'żenad', 'zenad',
			'syf',
		];

		foreach ($negativeMarkers as $n) {
			if ($n !== '' && mb_strpos($norm, $n) !== false) {
				return true;
			}
		}

		return false;
	}
}
