<?php
if (!defined('ABSPATH')) {
	exit;
}

final class ACWPCommentProtectionAjax {
	private const AJAX_ACTION = 'acwp_comment_submit';
	private const NONCE_ACTION = 'acwp_comment_submit_nonce';

	public static function init(): void {
		add_action('wp_ajax_' . self::AJAX_ACTION, [__CLASS__, 'handle']);
		add_action('wp_ajax_nopriv_' . self::AJAX_ACTION, [__CLASS__, 'handle']);
	}

	public static function getAjaxAction(): string {
		return self::AJAX_ACTION;
	}

	public static function getNonceAction(): string {
		return self::NONCE_ACTION;
	}

	public static function handle(): void {
		// nonce.
		check_ajax_referer(self::NONCE_ACTION, 'nonce');

		// dane formularza.
		$post = wp_unslash($_POST);

		// zalogowany = bez filtrów (zwykle zaufany).
		if (!ACWPCommentProtection::shouldBypassAllFilters()) {
			if (!ACWPCommentProtection::isAllowedForGuest($post)) {
				// minimalne rozróżnienie dla UI (CSS/komunikaty).
				$reason = 'spam';
				if (method_exists('ACWPCommentProtection', 'getBlockReasonForGuest')) {
					$reason = (string) ACWPCommentProtection::getBlockReasonForGuest($post);
					if ($reason !== 'url') {
						$reason = 'spam';
					}
				}

				self::jsonOut([
					'ok'     => false,
					'state'  => 'rejected',
					'reason' => $reason, // 'url' albo 'spam'
				], 200);
			}
		}

		// zapis komentarza przez mechanizm WP (moderacja i hooki działają normalnie).
		if (!function_exists('wp_handle_comment_submission')) {
			require_once ABSPATH . 'wp-includes/comment.php';
		}

		$result = wp_handle_comment_submission($post);

		if (is_wp_error($result)) {
			// nie zdradzamy szczegółów – traktujemy jako spam/odrzut.
			self::jsonOut([
				'ok'     => false,
				'state'  => 'rejected',
				'reason' => 'spam',
			], 200);
		}

		$state = 'pending';
		if ($result instanceof WP_Comment) {
			$approved = (string) $result->comment_approved;
			$state = ($approved === '1') ? 'approved' : 'pending';
		}

		self::jsonOut([
			'ok'    => true,
			'state' => $state,
		], 200);
	}

	private static function jsonOut(array $data, int $status): void {
		status_header($status);
		nocache_headers();
		wp_send_json($data);
	}
}
