<?php
if (!defined('ABSPATH')) {
	exit;
}

final class ACWPCommentProtectionAdmin {
	public static function init(): void {
		// tworzymy osobną sekcję w Ustawienia → Dyskusja, żeby ustawienia wtyczki były czytelne i wyróżnione.
		add_settings_section(
			'acwp_comment_protection',
			'Comment Protection – Antyspam komentarzy',
			[__CLASS__, 'renderSectionInfo'],
			'discussion'
		);

		// podbijamy UX — po wejściu z linku z kotwicą przewijamy i podświetlamy tę sekcję.
		add_action('admin_enqueue_scripts', [__CLASS__, 'enqueueAdminAssets']);

		// pola ustawień dokładamy do Ustawienia → Dyskusja.
		// 1) HARD (agresywne) - kompatybilne ze starą opcją.
		register_setting('discussion', ACWPCommentProtection::OPTION_HARD_TERMS, [
			'type'              => 'string',
			'sanitize_callback' => [__CLASS__, 'sanitizeListValue'],
			'default'           => '',
		]);

		add_settings_field(
			ACWPCommentProtection::OPTION_HARD_TERMS,
			'Twarde zakazane tokeny (agresywne)',
			[__CLASS__, 'renderHardTermsField'],
			'discussion',
			'acwp_comment_protection'
		);

		// 2) SOFT (dotychczasowa logika).
		register_setting('discussion', ACWPCommentProtection::OPTION_SOFT_TERMS, [
			'type'              => 'string',
			'sanitize_callback' => [__CLASS__, 'sanitizeListValue'],
			'default'           => '',
		]);

		add_settings_field(
			ACWPCommentProtection::OPTION_SOFT_TERMS,
			'Miękkie słowa/frazy (inteligentne dopasowanie)',
			[__CLASS__, 'renderSoftTermsField'],
			'discussion',
			'acwp_comment_protection'
		);

		// 3) Domeny e-mail.
		register_setting('discussion', ACWPCommentProtection::OPTION_BLOCKED_EMAIL_SUFFIXES, [
			'type'              => 'string',
			'sanitize_callback' => [__CLASS__, 'sanitizeListValue'],
			'default'           => '',
		]);

		add_settings_field(
			ACWPCommentProtection::OPTION_BLOCKED_EMAIL_SUFFIXES,
			'Blokada domen e-mail (np. .ru; mail.ru)',
			[__CLASS__, 'renderEmailSuffixesField'],
			'discussion',
			'acwp_comment_protection'
		);
	}

	public static function renderSectionInfo(): void {
		// dodajemy kotwicę, żeby dało się przewinąć z listy wtyczek.
		echo '<div id="acwp-comment-protection"></div>';
		echo '<p class="description">'
			. esc_html__('Sekcja wtyczki Comment Protection. Ustawienia działają dla domyślnego formularza komentarzy WordPress (większość motywów).', 'comment-protection')
			. '</p>';
		echo '<p class="description">'
			. esc_html__('Wartości wpisuj jako listę rozdzieloną średnikami. Zostaw puste pole, aby wyłączyć daną regułę.', 'comment-protection')
			. '</p>';
	}

	public static function enqueueAdminAssets(string $hook_suffix): void {
		if ($hook_suffix !== 'options-discussion.php') return;

		$handle = 'acwp-comment-protection-admin';
		wp_register_style($handle, false, [], ACWPCommentProtectionPlugin::VERSION);
		wp_enqueue_style($handle);
		wp_add_inline_style($handle, '
			#acwp-comment-protection{position:relative;top:-64px;}
			.acwp-cp-highlight-table{box-shadow:0 0 0 2px rgba(0,0,0,.08);}
			.acwp-cp-highlight-title{box-shadow:0 0 0 2px rgba(0,0,0,.08);padding:6px 10px;display:inline-block;}
		');

		wp_register_script($handle, '', [], ACWPCommentProtectionPlugin::VERSION, true);
		wp_enqueue_script($handle);
		wp_add_inline_script($handle, "(function(){\n\ttry{\n\t\tif(!location.hash || location.hash !== '#acwp-comment-protection') return;\n\t\tvar anchor=document.getElementById('acwp-comment-protection');\n\t\tif(!anchor) return;\n\t\tanchor.scrollIntoView({block:'start'});\n\t\t// podświetlenie tytułu sekcji\n\t\tvar p=anchor;\n\t\twhile(p && p.tagName!=='H2'){p=p.previousElementSibling;}\n\t\tif(p) p.classList.add('acwp-cp-highlight-title');\n\t\t// podświetlenie tabeli ustawień (najbliższa tabela po kotwicy)\n\t\tvar t=anchor;\n\t\twhile(t && t.tagName!=='TABLE'){t=t.nextElementSibling;}\n\t\tif(t) t.classList.add('acwp-cp-highlight-table');\n\t}catch(e){}\n})();");
	}

	public static function sanitizeListValue($value): string {
		$value = (string)$value;
		// wycinamy kontrolne, zostawiamy tekst i średniki.
		$value = preg_replace('/[[:cntrl:]]/u', ' ', $value);
		$value = preg_replace('/\s+/u', ' ', $value);
		return trim($value);
	}

	public static function renderHardTermsField(): void {
		$value = (string)get_option(ACWPCommentProtection::OPTION_HARD_TERMS, '');
		?>
		<textarea
			name="<?php echo esc_attr(ACWPCommentProtection::OPTION_HARD_TERMS); ?>"
			id="<?php echo esc_attr(ACWPCommentProtection::OPTION_HARD_TERMS); ?>"
			rows="6"
			cols="60"
			class="large-text"
		><?php echo esc_textarea($value); ?></textarea>
		<p class="description">
			Tokeny rozdzielone średnikiem. Dopasowanie agresywne (np. <code>h t t p</code>, <code>h.t.t.p</code> też złapie <code>http</code>).
			Przykład: <code>http; https; hxxp; www; url</code>
		</p>
		<?php
	}

	public static function renderSoftTermsField(): void {
		$value = (string)get_option(ACWPCommentProtection::OPTION_SOFT_TERMS, '');
		?>
		<textarea
			name="<?php echo esc_attr(ACWPCommentProtection::OPTION_SOFT_TERMS); ?>"
			id="<?php echo esc_attr(ACWPCommentProtection::OPTION_SOFT_TERMS); ?>"
			rows="10"
			cols="60"
			class="large-text"
		><?php echo esc_textarea($value); ?></textarea>
		<p class="description">
			Frazy rozdzielone średnikiem. Działa „inteligentne” dopasowanie (granice słów, wildcard <code>*</code>).
			Przykład: <code>promocja; telegram*; t.me</code>
		</p>
		<?php
	}

	public static function renderEmailSuffixesField(): void {
		$value = (string)get_option(ACWPCommentProtection::OPTION_BLOCKED_EMAIL_SUFFIXES, '');
		?>
		<textarea
			name="<?php echo esc_attr(ACWPCommentProtection::OPTION_BLOCKED_EMAIL_SUFFIXES); ?>"
			id="<?php echo esc_attr(ACWPCommentProtection::OPTION_BLOCKED_EMAIL_SUFFIXES); ?>"
			rows="4"
			cols="60"
			class="large-text"
		><?php echo esc_textarea($value); ?></textarea>
		<p class="description">
			Końcówki domen e-mail rozdzielone średnikiem. Jeśli autor poda e-mail z taką domeną — komentarz odrzucony.
			Przykład: <code>.ru; mail.ru; .com</code>
		</p>
		<?php
	}
}