(function () {
  function onReady(fn) {
    if (document.readyState === 'loading') document.addEventListener('DOMContentLoaded', fn);
    else fn();
  }

  function $(sel) {
    return document.querySelector(sel);
  }

  function getVariantClass(variant) {
    if (variant === 'success') return 'comment-protection-notice--success';
    if (variant === 'url') return 'comment-protection-notice--url';
    return 'comment-protection-notice--error';
  }

  function getAriaRole(variant) {
    return variant === 'success' ? 'status' : 'alert';
  }

  function showInfoBox(text, respondSelector, variant) {
    var respond = $(respondSelector) || $('#respond');
    if (!respond) return null;

    // usuń poprzedni box, jeśli istnieje
    var old = respond.parentNode.querySelector('.comment-protection-notice');
    if (old) old.remove();

    var box = document.createElement('div');
    box.className = 'comment-protection-notice ' + getVariantClass(variant);
    box.setAttribute('role', getAriaRole(variant));
    box.textContent = text;

    // żeby dało się "bezpiecznie" fokusować (np. dla czytników ekranowych)
    box.setAttribute('tabindex', '-1');

    respond.parentNode.insertBefore(box, respond);
    return box;
  }

  // przewijaj do komunikatu (jeśli jest), a jak nie ma — do formularza
  function scrollToNoticeOrRespond(respondSelector) {
    var respond = $(respondSelector) || $('#respond');
    if (!respond) return;

    var box = respond.parentNode.querySelector('.comment-protection-notice');
    var el = box || respond;

    // 15% luzu od góry ekranu
    var offset = Math.round(window.innerHeight * 0.15);

    var rect = el.getBoundingClientRect();
    var targetTop = window.pageYOffset + rect.top - offset;

    // zabezpieczenie, żeby nie próbować scrollować "powyżej strony"
    if (targetTop < 0) targetTop = 0;

    window.scrollTo({ top: targetTop, behavior: 'smooth' });

    // Fokus na komunikacie (bez dodatkowego scrolla)
    if (box && typeof box.focus === 'function') {
      try { box.focus({ preventScroll: true }); } catch (e) { box.focus(); }
    }
  }

  function hideWebsiteField() {
    var urlInput = document.getElementById('url') || document.querySelector('input[name="url"]');
    if (!urlInput) return;

    var wrap =
      urlInput.closest('.comment-form-url') ||
      urlInput.closest('.comment-form-field-input-url') ||
      urlInput.closest('p') ||
      null;

    var el = wrap || urlInput;
    el.style.position = 'absolute';
    el.style.left = '-9999px';
    el.style.top = 'auto';
    el.style.width = '1px';
    el.style.height = '1px';
    el.style.overflow = 'hidden';
  }

  function makeEmailOptional() {
    var email = document.getElementById('email') || document.querySelector('input[name="email"]');
    if (!email) return;
    email.removeAttribute('required');
    email.removeAttribute('aria-required');
  }

  function ensureDefaultAuthor(defaultAuthor) {
    var author = document.getElementById('author') || document.querySelector('input[name="author"]');
    if (!author) return;
    if (!author.value.trim()) author.value = defaultAuthor;
    author.setAttribute('required', 'required');
    author.setAttribute('aria-required', 'true');
  }

  function pickMessage(cfg, variant) {
    // zachowujemy kompatybilność ze starym cfg.message
    var legacy = cfg.message || 'Komentarz dodany i niedługo pojawi się na stronie';

    if (variant === 'success') return cfg.successMessage || legacy;
    if (variant === 'url') return cfg.urlMessage || 'Nasze systemy wykryły link w komentarzu i został odrzucony.';
    return cfg.spamMessage || 'Nasze systemy antyspamowe wykryły podejrzaną treść i komentarz został odrzucony.';
  }

  function parseJsonSafe(res) {
    return res.json().catch(function () { return null; });
  }

  async function submitCommentAjax(form, cfg) {
    var fd = new FormData(form);
    fd.append('action', cfg.action);
    fd.append('nonce', cfg.nonce);

    // wyłącz przycisk submit na czas requestu
    var submitBtn = form.querySelector('input[type="submit"], button[type="submit"]');
    if (submitBtn) submitBtn.disabled = true;

    try {
      var res = await fetch(cfg.ajaxUrl, {
        method: 'POST',
        body: fd,
        credentials: 'same-origin',
        headers: { 'X-Requested-With': 'XMLHttpRequest' }
      });

      var data = await parseJsonSafe(res);

      // domyślnie sukces (zgodne z Twoim UX, ale teraz rozróżniamy przypadki)
      var variant = 'success';

      if (!data || data.ok !== true) {
        // jeśli serwer zwrócił reason=url → osobny komunikat
        if (data && data.reason === 'url') variant = 'url';
        else variant = 'error';
      }

      showInfoBox(pickMessage(cfg, variant), cfg.respondSelector, variant);
      scrollToNoticeOrRespond(cfg.respondSelector);

      // Opcjonalnie: czyścimy treść komentarza (zwykle tak jest po submit)
      var comment = document.getElementById('comment') || form.querySelector('textarea[name="comment"]');
      if (comment) comment.value = '';

      return res;
    } catch (e) {
      // błąd sieci → komunikat błędu (nie success)
      var variantErr = 'error';
      showInfoBox(pickMessage(cfg, variantErr), cfg.respondSelector, variantErr);
      scrollToNoticeOrRespond(cfg.respondSelector);
    } finally {
      if (submitBtn) submitBtn.disabled = false;
    }
  }

  onReady(function () {
    var cfg = window.commentProtectionCfg || {};
    if (!cfg.ajaxUrl || !cfg.action || !cfg.nonce) return;

    ensureDefaultAuthor(cfg.defaultAuthor || 'Anonimowy');
    makeEmailOptional();
    hideWebsiteField();

    // Blocksy może dynamicznie przerysować formularz
    var respond = document.getElementById('respond');
    if (respond && window.MutationObserver) {
      var obs = new MutationObserver(function () {
        ensureDefaultAuthor(cfg.defaultAuthor || 'Anonimowy');
        makeEmailOptional();
        hideWebsiteField();
      });
      obs.observe(respond, { childList: true, subtree: true });
    }

    // przechwyt submit (capture=true, żeby wygrać z innymi handlerami)
    document.addEventListener('submit', function (ev) {
      var form = ev.target;
      if (!form) return;

      var isCommentForm = false;
      if (cfg.formSelector && form.matches && form.matches(cfg.formSelector)) isCommentForm = true;
      if (!isCommentForm && form.id === 'commentform') isCommentForm = true;

      if (!isCommentForm) return;

      ev.preventDefault();
      ev.stopPropagation();

      submitCommentAjax(form, cfg);
    }, true);
  });
})();