<?php
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Komentarz po polsku: sanitizacja datetime-local: YYYY-MM-DDTHH:MM
 */
function maintenanceModeSanitizeDatetimeLocal($value): string {
    $value = trim((string) $value);
    if ($value === '') {
        return '';
    }
    return preg_match('/^\d{4}-\d{2}-\d{2}T\d{2}:\d{2}$/', $value) ? $value : '';
}

/**
 * Komentarz po polsku: whitelist IP jako tekst; normalizacja EOL + limit długości.
 */
function maintenanceModeSanitizeWhitelist($value): string {
    $value = (string) $value;
    $value = str_replace(["\r\n", "\r"], "\n", $value);
    return substr($value, 0, 5000);
}

/**
 * Komentarz po polsku: sanitizacja listy ról tylko do ról edytowalnych.
 */
function maintenanceModeSanitizeRoles($value): array {
    $editable = array_keys(get_editable_roles());
    $out = [];

    if (is_array($value)) {
        foreach ($value as $role) {
            $role = sanitize_key((string) $role);
            if (in_array($role, $editable, true)) {
                $out[] = $role;
            }
        }
    }

    // Domyślnie: administrator + editor
    if (empty($out)) {
        $out = ['administrator', 'editor'];
    }

    return array_values(array_unique($out));
}

/**
 * Komentarz po polsku: opis sekcji + kotwica do przewijania (options-general.php#maintenance_mode_section).
 * To daje UX 1:1 jak w Export to PDF.
 */
function maintenanceModeSectionDescription(): void {
    echo '<span id="' . esc_attr(MAINT_MODE_SETTINGS_ANCHOR) . '"></span>';
    // Jeśli chcesz, możesz tu dodać opis w przyszłości (na razie zostawiamy czysto).
}

/**
 * Rejestracja ustawień + pól (Ustawienia → Ogólne)
 */
function maintenanceModeRegisterSettings(): void {

    register_setting('general', MAINT_MODE_OPT_ENABLED, [
        'type'              => 'boolean',
        'sanitize_callback' => 'rest_sanitize_boolean',
        'default'           => false,
    ]);

    register_setting('general', MAINT_MODE_OPT_UNTIL, [
        'type'              => 'string',
        'sanitize_callback' => 'maintenanceModeSanitizeDatetimeLocal',
        'default'           => '',
    ]);

    register_setting('general', MAINT_MODE_OPT_ALLOWED_ROLES, [
        'type'              => 'array',
        'sanitize_callback' => 'maintenanceModeSanitizeRoles',
        'default'           => ['administrator', 'editor'],
    ]);

    register_setting('general', MAINT_MODE_OPT_IP_WHITELIST, [
        'type'              => 'string',
        'sanitize_callback' => 'maintenanceModeSanitizeWhitelist',
        'default'           => '',
    ]);

    register_setting('general', MAINT_MODE_OPT_TITLE, [
        'type'              => 'string',
        'sanitize_callback' => 'sanitize_text_field',
        'default'           => 'Tryb konserwacji',
    ]);

    register_setting('general', MAINT_MODE_OPT_CONTENT, [
        'type'              => 'string',
        'sanitize_callback' => 'wp_kses_post',
        'default'           => '',
    ]);

    register_setting('general', MAINT_MODE_OPT_BG, [
        'type'              => 'string',
        'sanitize_callback' => 'esc_url_raw',
        'default'           => '',
    ]);

    add_settings_section(
        'maintenance_mode_section', // WP internal id sekcji
        'Ustawienia trybu konserwacji',
        'maintenanceModeSectionDescription', // tu wstrzykujemy kotwicę
        'general'
    );

    add_settings_field(
        MAINT_MODE_OPT_ENABLED,
        'Włącz tryb konserwacji',
        'maintenanceModeFieldEnabled',
        'general',
        'maintenance_mode_section'
    );

    add_settings_field(
        MAINT_MODE_OPT_UNTIL,
        'Auto-wyłączenie (data i godzina)',
        'maintenanceModeFieldUntil',
        'general',
        'maintenance_mode_section'
    );

    add_settings_field(
        MAINT_MODE_OPT_ALLOWED_ROLES,
        'Dostęp w czasie konserwacji (role)',
        'maintenanceModeFieldRoles',
        'general',
        'maintenance_mode_section'
    );

    add_settings_field(
        MAINT_MODE_OPT_IP_WHITELIST,
        'Whitelist IP (zawsze wpuszcza)',
        'maintenanceModeFieldWhitelist',
        'general',
        'maintenance_mode_section'
    );

    add_settings_field(
        MAINT_MODE_OPT_TITLE,
        'Tytuł strony',
        'maintenanceModeFieldTitle',
        'general',
        'maintenance_mode_section'
    );

    add_settings_field(
        MAINT_MODE_OPT_CONTENT,
        'Treść komunikatu',
        'maintenanceModeFieldContent',
        'general',
        'maintenance_mode_section'
    );

    add_settings_field(
        MAINT_MODE_OPT_BG,
        'Obraz tła',
        'maintenanceModeFieldBackground',
        'general',
        'maintenance_mode_section'
    );
}

/**
 * Pola ustawień
 */
function maintenanceModeFieldEnabled(): void {
    $enabled = (bool) get_option(MAINT_MODE_OPT_ENABLED, false);
    echo '<input type="checkbox" name="' . esc_attr(MAINT_MODE_OPT_ENABLED) . '" value="1" ' . checked(true, $enabled, false) . ' />';
}

function maintenanceModeFieldUntil(): void {
    $value = (string) get_option(MAINT_MODE_OPT_UNTIL, '');
    echo '<input type="datetime-local" name="' . esc_attr(MAINT_MODE_OPT_UNTIL) . '" value="' . esc_attr($value) . '" />';
    echo '<p class="description">Po osiągnięciu tej daty wtyczka automatycznie wyłączy tryb konserwacji (czas WordPressa).</p>';
}

function maintenanceModeFieldRoles(): void {
    $selected = get_option(MAINT_MODE_OPT_ALLOWED_ROLES, ['administrator', 'editor']);
    if (!is_array($selected)) {
        $selected = ['administrator', 'editor'];
    }

    $roles = get_editable_roles();

    echo '<fieldset>';
    foreach ($roles as $key => $data) {
        $isChecked = in_array($key, $selected, true);
        echo '<label style="display:block;margin:4px 0;">';
        echo '<input type="checkbox" name="' . esc_attr(MAINT_MODE_OPT_ALLOWED_ROLES) . '[]" value="' . esc_attr($key) . '" ' . checked(true, $isChecked, false) . ' /> ';
        echo esc_html($data['name']);
        echo '</label>';
    }
    echo '<p class="description">Zaznaczone role widzą stronę normalnie w trakcie konserwacji.</p>';
    echo '</fieldset>';
}

function maintenanceModeFieldWhitelist(): void {
    $value = (string) get_option(MAINT_MODE_OPT_IP_WHITELIST, '');
    echo '<textarea name="' . esc_attr(MAINT_MODE_OPT_IP_WHITELIST) . '" rows="6" class="large-text code">' . esc_textarea($value) . '</textarea>';
    echo '<p class="description">Jedna reguła na linię. Obsługa: IP, CIDR (np. 1.2.3.0/24), zakres IPv4 (np. 1.2.3.4-1.2.3.20). Linie zaczynające się od # są ignorowane.</p>';
}

function maintenanceModeFieldTitle(): void {
    $value = (string) get_option(MAINT_MODE_OPT_TITLE, 'Tryb konserwacji');
    echo '<input type="text" name="' . esc_attr(MAINT_MODE_OPT_TITLE) . '" value="' . esc_attr($value) . '" class="regular-text" />';
}

function maintenanceModeFieldContent(): void {
    wp_editor(
        (string) get_option(MAINT_MODE_OPT_CONTENT, ''),
        'maintenance_mode_content',
        [
            'textarea_name' => MAINT_MODE_OPT_CONTENT,
            'textarea_rows' => 8,
            'media_buttons' => true,
            'teeny'         => false,
        ]
    );
}

function maintenanceModeFieldBackground(): void {
    $value = esc_attr((string) get_option(MAINT_MODE_OPT_BG, ''));
    ?>
    <input type="text" id="maintenance_mode_background_image" name="<?php echo esc_attr(MAINT_MODE_OPT_BG); ?>" value="<?php echo $value; ?>" class="regular-text" />
    <button type="button" class="button" id="maintenance_mode_background_image_button">Wybierz obraz</button>
    <script>
        jQuery(function($){
            let frame;
            $('#maintenance_mode_background_image_button').on('click', function(e){
                e.preventDefault();
                if (frame) { frame.open(); return; }
                frame = wp.media({
                    title: 'Wybierz obraz tła',
                    button: { text: 'Użyj obrazu' },
                    multiple: false
                });
                frame.on('select', function(){
                    const attachment = frame.state().get('selection').first().toJSON();
                    $('#maintenance_mode_background_image').val(attachment.url);
                });
                frame.open();
            });
        });
    </script>
    <?php
}