<?php
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Komentarz po polsku: pobiera IP klienta (wspiera Cloudflare / proxy).
 */
function maintenanceModeGetClientIp(): string {
    $candidates = [];

    if (!empty($_SERVER['HTTP_CF_CONNECTING_IP'])) {
        $candidates[] = (string) $_SERVER['HTTP_CF_CONNECTING_IP'];
    }

    if (!empty($_SERVER['HTTP_X_FORWARDED_FOR'])) {
        $xff = (string) $_SERVER['HTTP_X_FORWARDED_FOR'];
        $parts = array_map('trim', explode(',', $xff));
        if (!empty($parts[0])) {
            $candidates[] = $parts[0];
        }
    }

    if (!empty($_SERVER['REMOTE_ADDR'])) {
        $candidates[] = (string) $_SERVER['REMOTE_ADDR'];
    }

    foreach ($candidates as $ip) {
        if (filter_var($ip, FILTER_VALIDATE_IP)) {
            return $ip;
        }
    }

    return '';
}

/**
 * Komentarz po polsku: sprawdza czy IP jest na whiteliście (IP / CIDR / zakres IPv4).
 */
function maintenanceModeIpInWhitelist(string $ip, string $rawList): bool {
    $ip = trim($ip);
    if ($ip === '' || !filter_var($ip, FILTER_VALIDATE_IP)) {
        return false;
    }

    $lines = preg_split('/\R/u', (string) $rawList) ?: [];
    foreach ($lines as $line) {
        $line = trim($line);
        if ($line === '' || str_starts_with($line, '#')) {
            continue;
        }

        // Exact IP
        if (filter_var($line, FILTER_VALIDATE_IP)) {
            if ($line === $ip) {
                return true;
            }
            continue;
        }

        // CIDR
        if (preg_match('/^([0-9a-fA-F\:\.]+)\/(\d{1,3})$/', $line, $m)) {
            $subnet = $m[1];
            $mask = (int) $m[2];

            if (filter_var($subnet, FILTER_VALIDATE_IP, FILTER_FLAG_IPV4) && filter_var($ip, FILTER_VALIDATE_IP, FILTER_FLAG_IPV4)) {
                if ($mask < 0 || $mask > 32) continue;
                $ipLong = ip2long($ip);
                $subLong = ip2long($subnet);
                $maskLong = -1 << (32 - $mask);
                if (($ipLong & $maskLong) === ($subLong & $maskLong)) {
                    return true;
                }
                continue;
            }

            if (filter_var($subnet, FILTER_VALIDATE_IP, FILTER_FLAG_IPV6) && filter_var($ip, FILTER_VALIDATE_IP, FILTER_FLAG_IPV6)) {
                if ($mask < 0 || $mask > 128) continue;
                $subBin = inet_pton($subnet);
                $ipBin  = inet_pton($ip);
                if ($subBin === false || $ipBin === false) continue;

                $bytes = intdiv($mask, 8);
                $bits  = $mask % 8;

                if ($bytes > 0 && substr($subBin, 0, $bytes) !== substr($ipBin, 0, $bytes)) {
                    continue;
                }
                if ($bits === 0) {
                    return true;
                }
                $maskByte = chr((0xFF << (8 - $bits)) & 0xFF);
                if (((ord($subBin[$bytes]) & ord($maskByte)) === (ord($ipBin[$bytes]) & ord($maskByte)))) {
                    return true;
                }
            }

            continue;
        }

        // IPv4 range: a-b
        if (preg_match('/^(\d{1,3}(?:\.\d{1,3}){3})\s*-\s*(\d{1,3}(?:\.\d{1,3}){3})$/', $line, $m)) {
            $a = $m[1];
            $b = $m[2];
            if (!filter_var($a, FILTER_VALIDATE_IP, FILTER_FLAG_IPV4) || !filter_var($b, FILTER_VALIDATE_IP, FILTER_FLAG_IPV4)) {
                continue;
            }
            if (!filter_var($ip, FILTER_VALIDATE_IP, FILTER_FLAG_IPV4)) {
                continue;
            }
            $ipLong = ip2long($ip);
            $aLong = ip2long($a);
            $bLong = ip2long($b);
            if ($aLong !== false && $bLong !== false && $ipLong !== false) {
                $min = min($aLong, $bLong);
                $max = max($aLong, $bLong);
                if ($ipLong >= $min && $ipLong <= $max) {
                    return true;
                }
            }
        }
    }

    return false;
}

/**
 * Komentarz po polsku: sprawdza czy zalogowany user ma dostęp po rolach.
 */
function maintenanceModeUserHasRoleAccess(): bool {
    if (!is_user_logged_in()) {
        return false;
    }

    $allowed = get_option(MAINT_MODE_OPT_ALLOWED_ROLES, ['administrator', 'editor']);
    if (!is_array($allowed) || empty($allowed)) {
        $allowed = ['administrator', 'editor'];
    }

    $user = wp_get_current_user();
    if (!$user || empty($user->roles)) {
        return false;
    }

    foreach ($user->roles as $role) {
        if (in_array($role, $allowed, true)) {
            return true;
        }
    }

    return false;
}

/**
 * Komentarz po polsku: auto-wyłączenie po dacie (czas WP).
 */
function maintenanceModeMaybeAutoDisable(): void {
    if (!get_option(MAINT_MODE_OPT_ENABLED, false)) {
        return;
    }

    $until = trim((string) get_option(MAINT_MODE_OPT_UNTIL, ''));
    if ($until === '') {
        return;
    }

    $ts = strtotime(str_replace('T', ' ', $until));
    if ($ts === false) {
        return;
    }

    $now = (int) current_time('timestamp');
    if ($now >= $ts) {
        update_option(MAINT_MODE_OPT_ENABLED, false);
    }
}
