// assets/print-pdf.js
jQuery(function ($) {
  function getOptions() {
    var o = (window.exportToPDF && exportToPDF.options) ? exportToPDF.options : {};
    return {
      hideImages: !!parseInt(o.hideImages || 0, 10),
      hideLinks: !!parseInt(o.hideLinks || 0, 10),
      hideToc: !!parseInt(o.hideToc || 0, 10)
    };
  }

  function removeImages(root) {
    // usuwamy obrazki i typowe kontenery obrazków (figury).
    root.querySelectorAll("img, figure, picture, source").forEach(function (el) {
      el.remove();
    });
  }

  function removeLinks(root) {
    // zamieniamy <a> na zwykły tekst (żeby nie tracić treści).
    root.querySelectorAll("a").forEach(function (a) {
      var span = root.ownerDocument.createElement("span");
      span.textContent = a.textContent || "";
      a.replaceWith(span);
    });
  }

  function removeToc(root) {
    // heurystyka – wycinamy najczęstsze TOC (Gutenberg/Elementor/plug-iny).
    var selectors = [
      "nav.toc",
      "nav#toc",
      "div.toc",
      "div#toc",
      ".table-of-contents",
      ".toc-container",
      "nav.wp-block-table-of-contents",
      "ol.wp-block-table-of-contents",
      "ul.wp-block-table-of-contents",
      ".ez-toc-container",
      ".lwptoc",
      ".rank-math-toc",
      ".elementor-toc__container"
    ];

    selectors.forEach(function (sel) {
      root.querySelectorAll(sel).forEach(function (el) {
        el.remove();
      });
    });

    // dodatkowa heurystyka – elementy z tekstem „Spis treści” i linkami do #nagłówków.
    root.querySelectorAll("nav, div, section, aside").forEach(function (el) {
      var text = (el.innerText || "").trim().toLowerCase();
      if (!text) return;

      if (!(text.includes("spis treści") || text.includes("spis tresci") || text.includes("table of contents") || text === "spis")) {
        return;
      }

      var anchors = el.querySelectorAll('a[href^="#"]');
      if (anchors && anchors.length >= 3) {
        el.remove();
      }
    });
  }

  function applyPrintFilters(contentWrap) {
    var opt = getOptions();
    if (opt.hideToc) removeToc(contentWrap);
    if (opt.hideImages) removeImages(contentWrap);
    if (opt.hideLinks) removeLinks(contentWrap);
  }

  $(document).on("click", ".etp-print-link", function (e) {
    e.preventDefault();
    e.stopPropagation();

    var postId = parseInt($(this).data("post-id"), 10);
    if (!postId) return;

    if (!window.exportToPDF || !exportToPDF.ajaxUrl || !exportToPDF.nonce) {
      alert("ExportToPDF: missing config (exportToPDF).");
      return;
    }

    // otwieramy okno wcześniej (popup blockers).
    var win = window.open("", "_blank");
    if (!win) {
      alert((exportToPDF.i18n && exportToPDF.i18n.popupBlocked) || "Popup blocked.");
      return;
    }

    try {
      win.document.title = (exportToPDF.i18n && exportToPDF.i18n.loading) || "Loading...";
      win.document.body.innerHTML =
        "<p>" + ((exportToPDF.i18n && exportToPDF.i18n.loading) || "Loading...") + "</p>";
    } catch (err) {}

    $.ajax({
      url: exportToPDF.ajaxUrl,
      method: "POST",
      dataType: "json",
      data: {
        action: "export_to_pdf_fetch_post_content",
        post_id: postId,
        nonce: exportToPDF.nonce
      }
    })
      .done(function (response) {
        if (!response || !response.success || !response.data) {
          throw new Error("Invalid response");
        }

        var title = response.data.title || "";
        var contentHtml = response.data.content || "";

        var doc = win.document;
        doc.open();
        doc.write("<!doctype html><html><head><meta charset='utf-8'></head><body></body></html>");
        doc.close();

        doc.title = title;

        // podstawowe style dla wydruku.
        var style = doc.createElement("style");
        style.type = "text/css";
        style.appendChild(
          doc.createTextNode(
            "body{font-family:Arial, sans-serif; margin:20px;}" +
              "h1{font-size:24px; margin:0 0 16px;}" +
              ".etp-content img{max-width:100%; height:auto;}" +
              "@media print{a[href]:after{content:\"\";}}"
          )
        );
        doc.head.appendChild(style);

        var h1 = doc.createElement("h1");
        h1.textContent = title;

        var contentWrap = doc.createElement("div");
        contentWrap.className = "etp-content";
        // treść jest sanitizowana po stronie PHP (wp_kses_post).
        contentWrap.innerHTML = contentHtml;

        // filtry wg ustawień (bez obrazków/linków/spisu treści).
        applyPrintFilters(contentWrap);

        doc.body.innerHTML = "";
        doc.body.appendChild(h1);
        doc.body.appendChild(contentWrap);

        win.focus();
        setTimeout(function () {
          win.print();
        }, 150);
      })
      .fail(function () {
        try {
          win.document.body.innerHTML =
            "<p>" +
            ((exportToPDF.i18n && exportToPDF.i18n.errorGeneric) || "An error occurred.") +
            "</p>";
        } catch (err) {}
      });
  });
});