<?php
/**
 * Plugin Name: Export to PDF
 * Description: Dodaje w WP-Admin link „Export to PDF” dla wpisów i stron (drukowanie przez przeglądarkę: Print → Save as PDF).
 * Version: 1.2.1
 * Requires at least: 6.0
 * Tested up to: 6.9
 * Requires PHP: 7.4
 * Author: ACWP (Mateusz Turek)
 * Author URI: https://acwp.pl/
 * License: GPLv2 or later
 * Text Domain: export-to-pdf
 */

if (!defined('ABSPATH')) {
    exit;
}

final class ExportToPDFPlugin
{
    private const NONCE_ACTION   = 'export_to_pdf_nonce';
    private const NONCE_FIELD    = 'nonce';
    private const AJAX_ACTION    = 'export_to_pdf_fetch_post_content';
    private const SCRIPT_HANDLE  = 'export-to-pdf-admin';

    private const OPTION_KEY     = 'export_to_pdf_options';
    private const SETTINGS_ANCHOR_ID = 'export_to_pdf_section';

    public static function init(): void
    {
        add_action('plugins_loaded', [__CLASS__, 'loadTextDomain']);

        add_filter('post_row_actions', [__CLASS__, 'addRowAction'], 10, 2);
        add_filter('page_row_actions', [__CLASS__, 'addRowAction'], 10, 2);

        add_action('admin_enqueue_scripts', [__CLASS__, 'enqueueAdminAssets']);

        add_action('wp_ajax_' . self::AJAX_ACTION, [__CLASS__, 'ajaxFetchPostContent']);

        add_action('admin_init', [__CLASS__, 'registerSettings']);

        // link „Ustawienia” na liście wtyczek.
        add_filter('plugin_action_links_' . plugin_basename(__FILE__), [__CLASS__, 'addSettingsActionLink']);

        // dodatkowy link w meta (tam gdzie „Wersja | Autor”).
        add_filter('plugin_row_meta', [__CLASS__, 'addPluginRowMetaLink'], 10, 2);
    }

    public static function loadTextDomain(): void
    {
        // przygotowanie pod tłumaczenia (jeśli kiedyś dodasz pliki .mo).
        load_plugin_textdomain('export-to-pdf', false, dirname(plugin_basename(__FILE__)) . '/languages');
    }

    public static function getOptions(): array
    {
        $defaults = [
            'hide_images' => 0,
            'hide_links'  => 0,
            'hide_toc'    => 0,
        ];

        $opt = get_option(self::OPTION_KEY);
        if (!is_array($opt)) {
            $opt = [];
        }

        $opt = array_merge($defaults, $opt);

        // normalizacja na 0/1.
        foreach ($defaults as $k => $_) {
            $opt[$k] = !empty($opt[$k]) ? 1 : 0;
        }

        return $opt;
    }

    public static function addRowAction(array $actions, WP_Post $post): array
    {
        // link pokazujemy tylko wtedy, gdy użytkownik może edytować dany wpis/stronę.
        if (!current_user_can('edit_post', (int) $post->ID)) {
            return $actions;
        }

        // bez JS kliknięcie nic nie robi (brak „#” w adresie).
        $actions['export_to_pdf'] = sprintf(
            '<a href="javascript:void(0)" class="etp-print-link" data-post-id="%d">%s</a>',
            (int) $post->ID,
            esc_html__('Export to PDF', 'export-to-pdf')
        );

        return $actions;
    }

    public static function enqueueAdminAssets(string $hookSuffix): void
    {
        // ładujemy skrypt wyłącznie na liście wpisów/stron (edit.php).
        if ($hookSuffix !== 'edit.php') {
            return;
        }

        $pluginUrl  = plugin_dir_url(__FILE__);
        $scriptRel  = 'includes/print-pdf.js';
        $scriptPath = plugin_dir_path(__FILE__) . $scriptRel;

        // cache-busting po mtime, żeby przeglądarka nie trzymała starego JS.
        $ver = is_file($scriptPath) ? (string) filemtime($scriptPath) : '1.2.1';

        wp_enqueue_script(
            self::SCRIPT_HANDLE,
            $pluginUrl . $scriptRel, // <-- WAŻNE: includes/print-pdf.js
            ['jquery'],
            $ver,
            true
        );

        $options = self::getOptions();

        wp_localize_script(self::SCRIPT_HANDLE, 'exportToPDF', [
            'ajaxUrl' => admin_url('admin-ajax.php'),
            'nonce'   => wp_create_nonce(self::NONCE_ACTION),
            'options' => [
                'hideImages' => (int) $options['hide_images'],
                'hideLinks'  => (int) $options['hide_links'],
                'hideToc'    => (int) $options['hide_toc'],
            ],
            'i18n' => [
                'loading'      => __('Loading...', 'export-to-pdf'),
                'errorGeneric' => __('An error occurred. Please try again.', 'export-to-pdf'),
                'popupBlocked' => __('Popup blocked by the browser. Please allow popups for this site.', 'export-to-pdf'),
            ],
        ]);
    }

    public static function ajaxFetchPostContent(): void
    {
        // bezpieczna weryfikacja nonce dla AJAX.
        check_ajax_referer(self::NONCE_ACTION, self::NONCE_FIELD);

        $postId = isset($_POST['post_id']) ? (int) $_POST['post_id'] : 0;
        if ($postId <= 0) {
            wp_send_json_error(['message' => 'Invalid post ID'], 400);
        }

        // twarda kontrola uprawnień do konkretnego posta.
        if (!current_user_can('edit_post', $postId)) {
            wp_send_json_error(['message' => 'Forbidden'], 403);
        }

        $post = get_post($postId);
        if (!$post instanceof WP_Post) {
            wp_send_json_error(['message' => 'Post not found'], 404);
        }

        // ograniczamy do post/page (zmień, jeśli chcesz wspierać CPT).
        if (!in_array((string) $post->post_type, ['post', 'page'], true)) {
            wp_send_json_error(['message' => 'Unsupported post type'], 400);
        }

        // przygotowanie treści jak na froncie (shortcode, oEmbed itd.).
        $content = apply_filters('the_content', (string) $post->post_content);

        // minimalna sanitizacja HTML, żeby ograniczyć ryzyko XSS w oknie wydruku.
        $content = wp_kses_post($content);

        $title = get_the_title($postId);

        wp_send_json_success([
            'title'   => (string) $title,
            'content' => $content,
        ]);
    }

    public static function registerSettings(): void
    {
        // rejestracja opcji (Ustawienia → Ogólne).
        register_setting(
            'general',
            self::OPTION_KEY,
            [
                'type'              => 'array',
                'sanitize_callback' => [__CLASS__, 'sanitizeOptions'],
                'default'           => self::getOptions(),
            ]
        );

        add_settings_section(
            self::SETTINGS_ANCHOR_ID,
            __('Export to PDF', 'export-to-pdf'),
            function (): void {
                // anchor, żeby link „Ustawienia” przewijał do sekcji.
                echo '<a id="' . esc_attr(self::SETTINGS_ANCHOR_ID) . '"></a>';
                echo '<p>' . esc_html__('Ustawienia wydruku dla akcji „Export to PDF” w WP-Admin.', 'export-to-pdf') . '</p>';
            },
            'general'
        );

        add_settings_field(
            'export_to_pdf_hide_images',
            __('Drukuj bez obrazków', 'export-to-pdf'),
            [__CLASS__, 'renderCheckboxField'],
            'general',
            self::SETTINGS_ANCHOR_ID,
            ['key' => 'hide_images']
        );

        add_settings_field(
            'export_to_pdf_hide_links',
            __('Drukuj bez linków', 'export-to-pdf'),
            [__CLASS__, 'renderCheckboxField'],
            'general',
            self::SETTINGS_ANCHOR_ID,
            ['key' => 'hide_links']
        );

        add_settings_field(
            'export_to_pdf_hide_toc',
            __('Drukuj bez spisu treści', 'export-to-pdf'),
            [__CLASS__, 'renderCheckboxField'],
            'general',
            self::SETTINGS_ANCHOR_ID,
            ['key' => 'hide_toc']
        );
    }

    public static function sanitizeOptions($value): array
    {
        // twarda sanitacja na 0/1.
        $value = is_array($value) ? $value : [];

        return [
            'hide_images' => !empty($value['hide_images']) ? 1 : 0,
            'hide_links'  => !empty($value['hide_links']) ? 1 : 0,
            'hide_toc'    => !empty($value['hide_toc']) ? 1 : 0,
        ];
    }

    public static function renderCheckboxField(array $args): void
    {
        $key = isset($args['key']) ? (string) $args['key'] : '';
        if ($key === '') {
            return;
        }

        $opt = self::getOptions();
        $checked = !empty($opt[$key]);

        $name = self::OPTION_KEY . '[' . $key . ']';

        printf(
            '<label><input type="checkbox" name="%s" value="1" %s> %s</label>',
            esc_attr($name),
            checked($checked, true, false),
            esc_html__('Włącz', 'export-to-pdf')
        );
    }

    public static function addSettingsActionLink(array $links): array
    {
        // link do Ustawienia → Ogólne (na anchor sekcji).
        $url = admin_url('options-general.php#' . self::SETTINGS_ANCHOR_ID);

        $settingsLink = sprintf(
            '<a href="%s">%s</a>',
            esc_url($url),
            esc_html__('Ustawienia', 'export-to-pdf')
        );

        array_unshift($links, $settingsLink);
        return $links;
    }

    public static function addPluginRowMetaLink(array $meta, string $file): array
    {
        if ($file !== plugin_basename(__FILE__)) {
            return $meta;
        }

        // link w „meta” wtyczki (tam gdzie „Wersja | Autor”).
        $url = admin_url('options-general.php#' . self::SETTINGS_ANCHOR_ID);

        $meta[] = sprintf(
            '<a href="%s">%s</a>',
            esc_url($url),
            esc_html__('Ustawienia', 'export-to-pdf')
        );

        return $meta;
    }
}

ExportToPDFPlugin::init();